<?php

namespace App\Http\Controllers\Dashboard\Configuraciones\Permisos;

use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Models\c_categorias_permiso;
use App\Models\c_sub_categorias_permiso;
use Spatie\Permission\Models\Permission;
use Stevebauman\Location\Facades\Location;

class PermisosController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function logRegistro($request, $status, $mensaje)
    {
        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permiso' => $request->all()
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Registro de permiso del sistema exitoso");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Registro de permiso del sistema fallido");
        }
    }

    public function logActulizacion($request, $status, $permiso, $mensaje)
    {
        $datosPermiso = $permiso == null ? $request->all() : $permiso;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permisos' => $datosPermiso
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Actualizacion de permiso del sistema exitoso");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Actualizacion de permiso del sistema fallido");
        }
    }

    public function logEliminacion($request, $status, $permiso, $mensaje)
    {
        $datosPermiso = $permiso == null ? $request->id_permiso_destroy : $permiso;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permisos' => $datosPermiso
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Eliminacion de permiso del sistema exitosa");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Eliminacion de permiso del sistema fallida");
        }
    }

    public function show()
    {
        $categorias = c_categorias_permiso::where('activo', 1)->get();

        return view('dashboard.configuraciones.permisos.permisos', compact('categorias'));
    }

    public function permisosData()
    {
        $data = DB::table('permissions')
            ->join('c_categorias_permisos', 'permissions.id_category', '=', 'c_categorias_permisos.id')
            ->join('c_sub_categorias_permisos', 'permissions.id_sub_category', '=', 'c_sub_categorias_permisos.id')
            ->select('permissions.id', 'permissions.name', 'permissions.description', 'permissions.id_category', 'permissions.id_sub_category', 'c_categorias_permisos.categoria', 'c_sub_categorias_permisos.sub_categoria')
            ->get();

        return DataTables::of($data)
            ->addColumn('action', function ($permiso) {
                return view('layouts.dashboard.partials.permisos.botones', compact('permiso'));
            })->make(true);
    }

    public function permisoData($id)
    {
        //Se obtiene los datos del rol
        $permisoData = Permission::findOrFail($id);

        //Retorna los datos del rol y los permisos en formato json
        return Response()->json(["success" => true, 'data' => $permisoData]);
    }

    public function getSubCategorias($id)
    {
        $subCategorias = c_sub_categorias_permiso::where('activo', 1)->where('id_categoria', $id)->get();

        return Response()->json(["success" => true, 'data' => $subCategorias]);
    }

    public function registraPermiso(Request $request)
    {
        $validaNombrePermiso = Permission::where('name', $request->nombre_permiso)->count();

        if ($validaNombrePermiso >= 1) {
            $mensaje = 'El nombre del permiso ya se encuentra registrado';
            $this->logRegistro($request, 'error', $mensaje);
            notify()->error($mensaje, 'Registro de permisos');
        } else {

            Permission::create([
                'name' => $request->nombre_permiso,
                'description' => $request->descripcion_permiso,
                'id_category' => $request->categoria,
                'id_sub_category' => $request->subCategoria,
            ]);

            $mensaje = 'Permiso registrado correctamente';
            $this->logRegistro($request, 'ok', $mensaje);
            notify()->success($mensaje, 'Registro de permisos');
        }

        return redirect()->route('configuraciones.permisos');
    }

    public function actulizaPermiso(Request $request)
    {
        //Recuperamos el id del permiso
        $id = $request->id_permiso_edit;

        //Se obtiene el nombre del permiso
        $nombrePermiso = $request->nombre_permiso_edit;

        //Verifica si el nombre del permiso ya se encuentra registrado
        $validaNombrePermiso = Permission::where('name', $nombrePermiso)->where('id', '!=', $id)->count();

        if ($validaNombrePermiso >= 1) {
            $mensaje = 'El nombre del permiso ya se encuentra registrado';
            $this->logActulizacion($request, 'error', null, $mensaje);
            notify()->error($mensaje, 'Actualización de permisos');
        } else {
            //Recuperamos el permiso a actualizar
            $permiso = Permission::findOrFail($id);

            //Actualizamos los datos del permiso
            $permiso->name = $nombrePermiso;
            $permiso->description = $request->descripcion_permiso_edit;
            $permiso->id_category = $request->categoria_edit;
            $permiso->id_sub_category = $request->subCategoria_edit;
            $permiso->update();

            $mensaje = 'Permiso actualizado correctamente';
            $this->logActulizacion($request, 'ok', $permiso, $mensaje);
            notify()->success($mensaje, 'Actualización de permisos');
        }

        return redirect()->route('configuraciones.permisos');
    }

    public function eliminaPermiso(Request $request)
    {
        $id = $request->id_permiso_destroy;

        //Valida si el permiso no esta asignado a ningun rol
        $validaPermisoRol = DB::table('role_has_permissions')->where('permission_id', '=', $id)->count();

        if ($validaPermisoRol >= 1) {
            $mensaje = 'El permiso no puede ser eliminado ya que esta asignado a un rol';
            $this->logEliminacion($request, 'error', null, $mensaje);
            notify()->error($mensaje, 'Eliminación del permisos');
        } else {
            //Recuperamos el permiso a eliminar
            $permiso = Permission::findOrFail($id);
            $permiso->delete();

            $mensaje = 'Permiso eliminado correctamente';
            $this->logEliminacion($request, 'ok', $permiso, $mensaje);
            notify()->success($mensaje, 'Eliminación del permisos');
        }

        return redirect()->route('configuraciones.permisos');
    }
}
