<?php

namespace App\Http\Controllers\Dashboard\Movimientos;

use App\Models\Sucursale;
use App\Models\NumeroReceta;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;
use App\Models\RegistroMovimientos;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Stevebauman\Location\Facades\Location;

class AjusteMovimientosController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function show()
    {
        $listaSucursales = Sucursale::select('sucursal', 'nombre')->where('activa', 1)->get();

        return view('dashboard.movimientos.ajustes', compact('listaSucursales'));
    }

    public function dataMovimientos($sucursal, $fechaInicial, $fechaFinal)
    {
        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $sucursal : session('sucursalSession');

        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);
        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        $data = $tableRegistros->whereBetween('fecha', [$fechaInicial, $fechaFinal])->get();

        return DataTables::of($data)->addColumn('action', function ($movimiento) {
            return view('layouts.dashboard.partials.movimientos.ajustes.botones', compact('movimiento'));
        })->make(true);
    }


    public function getDataMovimiento($ID, $sucursal)
    {
        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $sucursal : session('sucursalSession');

        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);
        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        return $tableRegistros->find($ID);
    }

    public function updateEntrada(Request $request)
    {
        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $request->sucursal : session('sucursalSession');
        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);

        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        $actulizaEntrada = $tableRegistros->find($request->id_edit);
        $actulizaEntrada->denominacion_dist = strtoupper($request->denominacionDist);
        $actulizaEntrada->denominacion_gen = strtoupper($request->denominacionGen);
        $actulizaEntrada->presentacion = strtoupper($request->presentacion);
        $actulizaEntrada->fecha = strtoupper($request->fecha);
        $actulizaEntrada->procedencia = strtoupper($request->procedencia);
        $actulizaEntrada->domicilio = strtoupper($request->domicilio);
        $actulizaEntrada->factura = strtoupper($request->numFactura);
        $actulizaEntrada->cantidad_adquirida = $request->cantidad;
        $actulizaEntrada->saldo = $request->saldo;
        $actulizaEntrada->observaciones = $request->observaciones;
        $actulizaEntrada->update();

        $mensaje = "Registro de entrada del libro de antibiotico actualizado correctamente.";
        AjusteMovimientosController::logUpdate($request, $mensaje, $sucursal);

        return Response()->json(["success" => true,  "mensaje" => $mensaje]);
    }

    public function updateSalida(Request $request)
    {
        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $request->sucursal : session('sucursalSession');
        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);

        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        $actulizaSalida = $tableRegistros->find($request->id_edit);
        $actulizaSalida->denominacion_dist = strtoupper($request->denominacionDist);
        $actulizaSalida->denominacion_gen = strtoupper($request->denominacionGen);
        $actulizaSalida->presentacion = strtoupper($request->presentacion);
        $actulizaSalida->fecha = strtoupper($request->fecha);
        $actulizaSalida->nombre_pree = strtoupper($request->preescribe);
        $actulizaSalida->domicilio = strtoupper($request->domicilio);
        $actulizaSalida->cedula = strtoupper($request->cedula);
        $actulizaSalida->cantidad_vendida = $request->cantidad;
        $actulizaSalida->saldo = $request->saldo;
        $actulizaSalida->observaciones = $request->observaciones;
        $actulizaSalida->update();

        $mensaje = "Registro de salida del libro de antibiotico actualizado correctamente.";
        AjusteMovimientosController::logUpdate($request, $mensaje, $sucursal);

        return Response()->json(["success" => true,  "mensaje" => $mensaje]);
    }


    public function deleteRegistro(Request $request)
    {
        $tipoMovimiento = $request->movimiento;
        $idRegistro = $request->id;

        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $request->sucursal : session('sucursalSession');
        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);

        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        $datosRegistro = $tableRegistros->find($idRegistro);
        $SKUProducto = $datosRegistro->codigo_producto;

        $eliminaRegistro = $datosRegistro;
        $eliminaRegistro->delete();

        Log::info("Registro [" . $idRegistro . "] eliminado de la sucursal [" . $sucursal . "]");

        //Este bloque se encarga de actualizar el inventario de los productos
        if ($tipoMovimiento == 'ENTRADA') {

            //Se actualiza el saldo a la cantidad correcta
            DB::table($tablaSucursal)
                ->where('codigo_producto', $SKUProducto)
                ->where('id', '>', $idRegistro)
                ->update(['saldo' => DB::raw('saldo - ' . $datosRegistro->cantidad_adquirida)]);

            Log::info("Saldo actualizado del producto [" . $SKUProducto . "] de la sucursal [" . $sucursal . "] [ENTRADA]");
        } else if ($tipoMovimiento == 'SALIDA') {

            //Se actualiza el saldo a la cantidad correcta
            DB::table($tablaSucursal)
                ->where('codigo_producto', $SKUProducto)
                ->where('id', '>', $idRegistro)
                ->update(['saldo' => DB::raw('saldo + ' . $datosRegistro->cantidad_vendida)]);

            Log::info("Saldo actualizado del producto [" . $SKUProducto . "] de la sucursal [" . $sucursal . "] [SALIDA]");

            if ($datosRegistro->receta != 'P') {
                //Se corrigue el numero de receta
                DB::table($tablaSucursal)
                    ->where('tipo_movimiento', 'SALIDA')
                    ->where('receta', '!=', 'P')
                    ->where('id', '>', $idRegistro)
                    ->update(['receta' => DB::raw('receta - 1')]);

                Log::info("Numeros de receta recorridos de la sucursal [" . $sucursal . "] [SALIDA]");

                //Se actuliza el numero de receta en la tabla del consecutivo
                DB::table('control_numero_receta_general')
                    ->where('sucursal', strtolower($sucursal))
                    ->update(['numero_receta' => DB::raw('numero_receta - 1')]);

                Log::info("Numeros de receta actualizado en el control general de la [" . $sucursal . "] [SALIDA]");
            }
        }

        //Se recorren los numeros de registro
        DB::table($tablaSucursal)
            ->where('id', '>', $idRegistro)
            ->update(['id' => DB::raw('id - 1')]);

        Log::info("ID'S recorridos de la sucursal [" . $sucursal . "]");

        //Se realiza la correcion del autoincrement de la tabla
        DB::statement('ALTER TABLE ' . $tablaSucursal . ' AUTO_INCREMENT = 1');
        Log::info("Se recorre el autoincrement de la sucursal [" . $sucursal . "]");

        $mensaje = "El registro de " . strtolower($request->movimiento) . " del libro de antibiotico se ha eliminado correctamente.";
        AjusteMovimientosController::logDelete($request, $mensaje, $sucursal);

        return Response()->json(["success" => true,  "mensaje" => $mensaje]);
    }


    public function corrigeNumeroReceta(Request $request)
    {
        $numRecetaAnterior = 1;

        $sucursal = (session('sucursalSession') == 'TODOS' || session('rolSession') != 'USUARIO') ? $request->sucursal_receta : session('sucursalSession');

        $tablaSucursal = "registro_movimientos_" . strtolower($sucursal);
        $tableRegistros = new RegistroMovimientos();
        $tableRegistros->setTable($tablaSucursal);

        //Se obtienen todos los registros que que sean una SALIDA y que tengan numero de receta
        $consultaRegistros = $tableRegistros->select('id', 'receta')->where('tipo_movimiento', 'SALIDA')->where('receta', '!=', 'P')->get();

        foreach ($consultaRegistros as $registro) {
            //Se obtiene el numero de receta  y se almacena en una variable para compararlo con el siguiente
            $numReceta = $registro->receta;

            //Se omite el primer registro
            if ($numReceta != 1) {
                //Se verifica que el numero de receta sea consecutivo
                if ($numRecetaAnterior != $numReceta - 1) {

                    //Se obtiene el numero de receta correspondiente al registro
                    $auxNumReceta = $numRecetaAnterior + 1;

                    //Se actualiza el numero de receta en la DB
                    $actulizaNumReceta = $tableRegistros->find($registro->id);
                    $actulizaNumReceta->receta = strtoupper($auxNumReceta);
                    $actulizaNumReceta->update();

                    //Se obtiene el numero de receta  y se almacena en una variable para compararlo con el siguiente
                    $numRecetaAnterior = $auxNumReceta;

                    Log::info("Sucursal [" . $sucursal . "] | ID [" . $registro->id . "] | Anterior [" . $numReceta . "] | Nuevo [" . $numRecetaAnterior . "]");
                } else {
                    //Se obtiene el numero de receta  y se almacena en una variable para compararlo con el siguiente
                    $numRecetaAnterior = $registro->receta;
                }
            }
        }


        $tableNumReceta = new NumeroReceta();
        $numeroReceta = $tableNumReceta->select('id', 'numero_receta')->where('sucursal', $sucursal)->get();
        $actulizaNumero = $tableNumReceta->find($numeroReceta[0]->id);
        $actulizaNumero->numero_receta = $numRecetaAnterior;
        $actulizaNumero->update();

        $mensaje = "Numeros de receta de la sucursal " . $sucursal . " se corrigieron correctamente.";
        AjusteMovimientosController::logCorrigeNumReceta($request, $mensaje, $sucursal);

        return Response()->json(["success" => true,  "mensaje" => $mensaje]);
    }


    private static function logUpdate(Request $request, $mensaje, $sucursal)
    {
        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'id_registro' => $request->id_edit
        ];

        Log::withContext($arrayDatos)->info("Registro actulizado correctamente [" . $sucursal . "]");
    }

    private static function logDelete(Request $request, $mensaje, $sucursal)
    {
        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'id_registro' => $request->id
        ];

        Log::withContext($arrayDatos)->info("Proceso de eliminacion de registro exitoso [" . $sucursal . "]");
    }

    private static function logCorrigeNumReceta(Request $request, $mensaje, $sucursal)
    {
        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje
        ];

        Log::withContext($arrayDatos)->info("Proceso de correccion de numeros de recetas exitoso [" . $sucursal . "]");
    }
}
